<?php
/**
 * @package   com_zoo
 * @author    YOOtheme http://www.yootheme.com
 * @copyright Copyright (C) YOOtheme GmbH
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

/*
   Class: ElementItemCategory
       The item category element class
*/
class ElementItemCategory extends Element implements iSubmittable {

    protected $_categories;

    /*
        Function: hasValue
            Checks if the element's value is set.

       Parameters:
            $params - render parameter

        Returns:
            Boolean - true, on success
    */
    public function hasValue($params = array()) {
        $categories = $this->_item->getRelatedCategories(true);
        return !empty($categories);
    }

    /*
        Function: render
            Renders the element.

       Parameters:
            $params - render parameter

        Returns:
            String - html
    */
    public function render($params = array()) {

        $params = $this->app->data->create($params);
        $values = array();
        foreach ($this->_item->getRelatedCategories(true) as $category) {
            $values[] = $params->get('linked') ? '<a href="'.$this->app->route->category($category).'">'.$category->name.'</a>' : $category->name;
        }

        return $this->app->element->applySeparators($params->get('separated_by'), $values);
    }

    /*
       Function: edit
           Renders the edit form field.

       Returns:
           String - html
    */
    public function edit() {
        return null;
    }

    /*
        Function: renderSubmission
            Renders the element in submission.

       Parameters:
            $params - AppData submission parameters

        Returns:
            String - html
    */
    public function renderSubmission($params = array()) {
        if ($layout = $this->getLayout('submission.php')) {
            return $this->renderLayout($layout, compact('params'));
        }
    }


    protected function _categoryList($application, $options, $name, $attribs = null, $key = 'value', $text = 'text', $selected = NULL, $idtag = false, $translate = false, $category = 0, $prefix = '-&nbsp;', $spacer = '.&nbsp;&nbsp;&nbsp;', $indent = '&nbsp;&nbsp;') {

        // set options
        settype($options, 'array');
        reset($options);

        // get category tree list
        $list = $this->app->tree->buildList($category, $application->getCategoryTree(), array(), $prefix, $spacer, $indent);

        // create options
        foreach ($list as $category) {

            if($category->hasChildren()){

                $options[] = $this->app->html->_('select.option', $category->id, $category->treename,'value','text',true);

            } else {
                $options[] = $this->app->html->_('select.option', $category->id, $category->treename);
            }

        }


        return $this->app->html->_('zoo.genericlist', $options, $name, $attribs, $key, $text, $selected, $idtag, $translate);
    }





    /*
        Function: validateSubmission
            Validates the submitted element

       Parameters:
            $value  - AppData value
            $params - AppData submission parameters

        Returns:
            Array - cleaned value
    */
    public function validateSubmission($value, $params) {
        $primary = (int) $value->get('primary');
        $value = (array) $value->get('value');

        if (!$params->get('multiple', true) && count($value) > 1) {
            $value = array(array_shift($value));
        }

        $primary = !$params->get('primary', false) || empty($primary) || !in_array($primary, $value) ? @$value[0] : $primary;

        $categories = array_keys($this->_item->getApplication()->getCategories());
        foreach ($value as $id) {
            if (!in_array($id, $categories)) {
                throw new AppValidatorException('Please choose valid categories only');
            }
        }

        if ($params->get('required') && !count($value)) {
            throw new AppValidatorException('Please choose a category');
        }

        // connect to submission aftersave event
        $this->app->event->dispatcher->connect('submission:saved', array($this, 'aftersubmissionsave'));

        return compact('value', 'primary');
    }

    /*
        Function: afterSubmissionSave
            Callback after item submission is saved

        Returns:
            void
    */
    public function afterSubmissionSave()
    {
        if (!empty($this->_categories))
        {
            if (in_array('0', $this->app->category->getItemsRelatedCategoryIds($this->_item->id)))
            {
                $this->_categories[] = 0;
            }

            $category_id = $this->_categories[0];

            if ($category_id)
            {
                $category = $this->app->table->category->get($category_id);

                if ($category->parent)
                {
                    $this->_categories[] = $category->parent;
                }
            }

            $this->_categories = array_unique($this->_categories);

            $this->_saveCategoryItemRelations($this->_item, $this->_categories);
        }
    }


    protected function _saveCategoryItemRelations($item, $categories) {
        // check ACL
        if (!$item->canEdit()) {
            //return false;
        }

        //init vars
        $db = $this->app->database;

        if (!is_array($categories)) {
            $categories = array($categories);
        }

        // trigger an event to let 3rd party extend the category list
        $this->app->event->dispatcher->notify($this->app->event->create($item, 'item:beforeSaveCategoryRelations', array('categories' => &$categories)));

        $categories = array_unique($categories);

        // delete category to item relations
        $query = "DELETE FROM ".ZOO_TABLE_CATEGORY_ITEM
            ." WHERE item_id=".(int) $item->id;

        // execute database query
        $db->query($query);

        // Generate the sql query for the categories
        $query_string = '(%s,'.(int) $item->id.')';
        $category_strings = array();
        foreach ($categories as $category) {
            if (is_numeric($category)) {
                $category_strings[] = sprintf($query_string, $category);
            }
        }

        // add category to item relations
        // insert relation to database
        if (!empty($category_strings)) {
            $query = "INSERT INTO ".ZOO_TABLE_CATEGORY_ITEM
                ." (category_id, item_id) VALUES ".implode(',', $category_strings);

            // execute database query
            $db->query($query);
        }

        $this->app->category->clearItemCategoryCache();

        return true;
    }


    /*
        Function: bindData
            Set data through data array.

        Parameters:
            $data - array

        Returns:
            Void
    */
    public function bindData($data = array())
    {
        $this->_categories = @$data['value'];
        $this->_item->getParams()->set('config.primary_category', @$data['primary']);
    }
}