<?php
/**
 * @package     Social Buttons Module
 * @subpackage  mod_social_buttons
 *
 * @copyright   Copyright (C) 2014 Yanovskiy Vitaliy All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;


/**
 * Class modSocialButtonsHelper
 */
class modSocialButtonsHelper
{

    /**
     * @var
     */
    protected $_app;

    /**
     * @var
     */
    protected $_document;

    /**
     * @var
     */
    protected $_input;

    /**
     * @var
     */
    public $name;


    /**
     * @param $params
     * @param $name
     * @return mixed
     */
    public function init($params, $name)
    {
        $this->_app = JFactory::getApplication();
        if(floatval(JVERSION) < 3){

            $this->_input    = array(
                'option' => $this->_app->input->get('option'),
                'view'   => $this->_app->input->get('view'),
                'id'     => $this->_app->input->get('id')
        );
            $this->_document = JFactory::getDocument();

        }else{
            $this->_document = $this->_app->getDocument();
            $this->_input    = $this->_app->input->getArray();
        }

        $this->name = $name;

        $this->_renderOpenGraph($params);

        $html['bookmarks'] = $this->_getBookmarks($params);
        $html['likes']     = $this->_renderLikes($params);
        $html['complex']   = $this->_renderComplex($params);

        return $html;
    }

    /**
     * Render bookmarks layout
     * @param array $params
     * @return null|string
     */
    protected function _getBookmarks($params)
    {

        // init vars
        $bookmarks = $this->_getBookmarkList();

        if ((int)$params->get('bookmark_enabled', 1) &&
            (int)$params->get('bookmarks_enabled', 1)
        ) {

            $this->_document->addStyleSheet('/modules/mod_social_buttons/assets/css/style-bookmarks.css');

            // get active jbzoobookmarks
            foreach ($bookmarks as $bookmark => $data) {
                if (!(int)$params->get('bookmark_' . $bookmark, 1)) {
                    unset($bookmarks[$bookmark]);
                }
            }

            return $bookmarks;
        } else {
            return null;
        }

    }

    /**
     * Render likes layout
     * @param array $params
     * @return string|null
     */
    protected function _renderLikes($params)
    {
        // init vars
        $pageId = '';

        if (isset($this->_input['id'])) {
            $pageId = $this->_input['id'];
        } elseif (isset($this->_input['item_id'])) {
            $pageId = $this->_input['item_id'];
        } else {
            $pageId = $this->_input['Itemid'];
        }

        $vkOptions = $fbOptions = $twOptions = $okOptions = $gpOptions = $liOptions = array();

        if ((int)$params->get('likes_enabled', 1)) {
            // Vkontakte Button
            if ((int)$params->get('like_vk_enabled', 1)) {

                $this->_document->addScript('http://vkontakte.ru/js/api/openapi.js');

                // get options like_vk button
                $vkOptions = array(
                    'vkEnabled' => $params->get('like_vk_enabled', 1),
                    'vkId'      => uniqid('like_vk'),
                    'id'        => $params->get('like_vk_id'),
                    'pageId'    => $pageId,
                    'params'    => array(
                        'type'      => str_replace('like_vk_', '', $params->get('like_vk_variants')),
                        'verb'      => $params->get('like_vk_verb'),
                        'height'    => $params->get('like_vk_height'),
                        'pageTitle' => $this->_document->getTitle(),
                        'pageUrl'   => JUri::current()
                    )
                );
            }

            // Facebook Button
            if ((int)$params->get('like_fb_enabled', 1)) {

                // get locale
                $lang = $params->get('like_fb_locale');

                // get string request
                $src = '//connect.facebook.net/' . $lang . '/all.js#xfbml=1';

                // get options FB like button
                $fbOptions = array(
                    'fbEnabled' => (int)$params->get('like_fb_enabled', 1),
                    'fbId'      => uniqid('fb-root'),
                    'class'     => 'fb-like',
                    'data-src'  => $src,
                    'params'    => 'class="fb-like" data-href="' . JUri::current() .
                        '" data-send="' . $params->get('like_fb_send') .
                        '" data-layout="' . $params->get('like_fb_layout') .
                        '" data-show-faces="' . $params->get('like_fb_show_faces') .
                        '" data-colorscheme="' . $params->get('like_fb_colorscheme') .
                        '" data-action="' . $params->get('like_fb_action') . '"'

                );
            }

            // Tweet Button
            if ((int)$params->get('like_tw_enabled', 1)) {

                // get options TW like button
                $twOptions = array(
                    'twEnabled' => (int)$params->get('like_tw_enabled', 1),
                    'params'    => 'href="https://twitter.com/share" class="twitter-share-button" data-url="' . JUri::current() .
                        '" data-lang="' . $params->get('like_tw_locale') . '" data-via="' . $params->get('like_tw_via') .
                        '" data-size="' . $params->get('like_tw_size') . '" data-related="' . $params->get('like_tw_related') .
                        '" data-hashtags="' . str_replace(' ', '', $params->get('like_tw_hashtags')) .
                        '" data-dnt="' . $params->get('like_tw_dnt') . '"'
                );

                if ((int)$params->get('like_tw_count', 1) == 0) {
                    $twOptions['params'] = $twOptions['params'] . ' data-count="none"';
                }
            }

            // Odnoklassniki button
            if ((int)$params->get('like_ok_enabled', 1)) {

                // get options OK like button
                $okOptions = array(
                    'okEnabled' => $params->get('like_ok_enabled'),
                    'okUrl'     => $params->get('like_ok_url'),
                    'okId'      => uniqid('ok_shareWidget'),
                    'params'    => array(
                        'nc' => $params->get('like_ok_count_enable'),
                        'vt' => $params->get('like_ok_count'),
                        'nt' => $params->get('like_ok_text'),
                        'ck' => $params->get('like_ok_ok_text'),
                        'st' => $params->get('like_ok_view'),
                        'sz' => (int)$params->get('like_ok_size')
                    ),

                );
            }

            // Google+ button
            if ((int)$params->get('like_gp_enabled', 1)) {

                // get options Google+ like button
                $gpOptions = array(
                    'gpEnabled'    => $params->get('like_gp_enabled'),
                    'scriptParams' => array(
                        'lang' => $params->get('like_gp_lang'),
                    ),
                    'params'       => 'class="g-plusone" data-size="' . $params->get('like_gp_size') .
                        '" data-annotation="' . $params->get('like_gp_annotation') .
                        '" data-width="' . $params->get('like_gp_width') . '"'
                );
            }

            // Linked In button
            if ((int)$params->get('like_li_enabled', 1)) {

                // get options Google+ like button
                $liOptions = array(
                    'liEnabled' => $params->get('like_li_enabled'),
                    'lang'      => $params->get('like_li_lang'),
                    'params'    => array(
                        'data-counter' => $params->get('like_li_counter'),
                        'data-url'     => $params->get('like_li_url')
                    )
                );
            }

            // render likes layout
            return array(
                'vkOption' => $vkOptions,
                'fbOption' => $fbOptions,
                'twOption' => $twOptions,
                'okOption' => $okOptions,
                'liOption' => $liOptions,
                'gpOption' => $gpOptions
            );

        }

        return null;
    }

    /**
     * Render complex layout
     * @param array $params
     * @return string|null
     */
    protected function _renderComplex($params)
    {
        $yaOptions   = array('element' => uniqid('ya_share'));
        $allServices = array(
            'blogger', 'delicious', 'diary', 'digg', 'evernote', 'facebook', 'friendfeed', 'gplus', 'juick',
            'liveinternet', 'linkedin', 'lj', 'moikrug', 'moimir', 'myspace', 'odnoklassniki', 'pinterest',
            'surfingbird', 'tutby', 'twitter', 'vkontakte', 'yaru', 'yazakladki'
        );

        if ((int)$params->get('like_ya_share_enabled', 1) &&
            (int)$params->get('complex_enabled', 1)
        ) {
            $yaOptions['yaEnabled'] = $params->get('like_ya_share_enabled');
            $this->_document->addScript('http://yandex.st/share/share.js');

            $services = JString::str_ireplace(' ', '', $params->get('like_ya_share_services'));
            if (!empty($services)) {
                $services = explode(',', $services);
            } else {
                $services = array('yaru', 'vkontakte', 'odnoklassniki', 'moimir', 'myspace', 'tutby', 'yazakladki');
            }

            $boxServices = array_values(array_diff($allServices, $services));

            $yaOptions['elementStyle'] = array(
                'type'          => $params->get('like_ya_share_style'),
                'quickServices' => $services
            );

            $yaOptions['title']       = $this->_document->getTitle();
            $yaOptions['description'] = $this->_replaceSpecial($this->_document->getDescription());
            $yaOptions['link']        = JUri::current();
            $yaOptions['image']       = '';
            $yaOptions['popupStyle']  = array('blocks' => $boxServices);
            $yaOptions['theme']       = $params->get('like_ya_share_theme', 'default');

            return array('yaOption' => $yaOptions);
        } else {
            return null;
        }
    }

    /**
     * @param $string
     * @return string
     */
    protected function _replaceSpecial($string)
    {
        $string = addslashes(str_replace('&nbsp;', ' ', strip_tags($string)));
        $string = str_replace(array("\r\n", "\r", "\n"), '', $string);
        $string = $this->_cutByWords($string, 250);

        return htmlspecialchars_decode($string, ENT_NOQUOTES);
    }

    /**
     * Get all available bookmarks
     * @return array $bookmarks
     */
    protected function _getBookmarkList()
    {
        // init var
        $bookmarks = array();

        // load xml file
        if ($xml = __DIR__ . '/bookmarks.xml') {
            $xml = @simplexml_load_file($xml);
            foreach ($xml as $bookmark => $list) {
                $bookmarks[(string)$list->attributes()->type] = array(
                    'label' => (string)$list->attributes()->name,
                    'link'  => (string)$list->link,
                    'click' => (string)$list->click
                );
            }
        }

        return $bookmarks;
    }

    /**
     * Get open graph tags
     * @return array
     */
    protected function _getOGData()
    {
        $db          = JFactory::getDbo();
        $description = $this->_document->getDescription();
        $paramsPage  = $this->_input;
        $id          = $paramsPage['id'];
        $ogTags      = array(
            'og:type'  => 'article',
            'og:title' => $this->_replaceSpecial($this->_document->getTitle()),
            'og:url'   => JUri::current(),
        );

        if (!empty($description)) {
            $ogTags['og:description'] = $description;
        }

        if ($paramsPage['view'] == 'category') {
            $result         = $db->setQuery("SELECT params FROM #__categories WHERE id='" . $id . "'")->loadResult();
            $paramsCategory = json_decode($result);
            $imagePath      = JUri::base() . $paramsCategory->image;
        } else if ($paramsPage['view'] == 'article') {
            $result        = $db->setQuery("SELECT images FROM #__content WHERE id='" . $id . "'")->loadResult();
            $imagesArticle = json_decode($result);

            if ($imagesArticle->image_intro) {
                $imagePath = JUri::base() . $imagesArticle->image_intro;
            } else if ($imagesArticle->image_fulltext) {
                $imagePath = JUri::base() . $imagesArticle->image_fulltext;
            }
        }

        if (!empty($imagePath) && $imagePath !== JUri::base()) {
            $ogTags['og:image'] = $imagePath;
        } else {
            $ogTags['og:image'] = JUri::base() . 'modules/' . $this->name . '/assets/images/default.jpg';
        }

        return $ogTags;
    }

    /**
     * Render open graph meta tags
     * @param $params
     */
    protected function _renderOpenGraph($params)
    {
        static $isOGShow = true;
        $paramsPage = $this->_input;

        if ($paramsPage['option'] == 'com_content' && $paramsPage['view'] == 'category' || $paramsPage['view'] == 'article') {
            $ogTags = $this->_getOGData();

            if ($isOGShow == true) {

                if ($params->get('like_vk_enabled') == '1' && $params->get('like_vk_id')) {
                    $ogTags['vk:app_id'] = $params->get('like_vk_id');
                }

                foreach ($ogTags as $key => $value) {
                    $value = strip_tags($value);
                    $value = str_replace(array("\n", "\r"), ' ', $value);
                    $value = htmlspecialchars_decode($value, ENT_NOQUOTES);
                    $value = $this->_cutByWords($value, 250);
                    $this->_document->addCustomTag('<meta property="' . $key . '" content="' . $value . '" />');
                }
            }

            $isOGShow = false;
        }
    }

    /**
     * Get truncated string (by words)
     * @param $string
     * @param $maxlen
     * @return string
     */
    protected function _cutByWords($string, $maxlen = 255)
    {

        $len    = (JString::strlen($string) > $maxlen) ? JString::strrpos(JString::substr($string, 0, $maxlen), ' ') : $maxlen;
        $cutStr = JString::substr($string, 0, $len);

        return (JString::strlen($string) > $maxlen) ? $cutStr . '...' : $cutStr;
    }

}