<?php
/**
 * JBZoo App is universal Joomla CCK, application for YooTheme Zoo component
 *
 * @package     jbzoo
 * @version     2.x Pro
 * @author      JBZoo App http://jbzoo.com
 * @copyright   Copyright (C) JBZoo.com,  All rights reserved.
 * @license     http://jbzoo.com/license-pro.php JBZoo Licence
 * @coder       Denis Smetannikov <denis@jbzoo.com>
 */

// no direct access
defined('_JEXEC') or die('Restricted access');


/**
 * Class JBModelElementJBPriceAdvance
 */
class JBModelElementJBPriceAdvance extends JBModelElement
{

    protected $_defaultCurrency = 'EUR';

    /**
     * Constructor
     * @param Element $element
     * @param $applicationId
     */
    function __construct(Element $element, $applicationId)
    {
        parent::__construct($element, $applicationId);

        $this->_defaultCurrency = $this->_config->get('currency_default', 'EUR');
    }

    /**
     * @param array|string $value
     * @param bool $exact
     * @return mixed|void
     */
    protected function _prepareValue($value, $exact = false)
    {
        if (is_array($value)) {
            $value = array_merge(array(
                'sku'        => '',
                'balance'    => '',
                'sale'       => '',
                'new'        => '',
                'val'        => '',
                'val_min'    => '',
                'val_max'    => '',
                'range'      => '',
                'currency'   => '',
                'val_type'   => 0,
                'price_type' => 0,
            ), $value);

            return $value;
        }

        return parent::_prepareValue($value, $exact);
    }

    /**
     * Set AND element conditions
     * @param JBDatabaseQuery $select
     * @param string $elementId
     * @param string|array $value
     * @param int $i
     * @param bool $exact
     * @return JBDatabaseQuery
     */
    public function conditionAND(JBDatabaseQuery $select, $elementId, $value, $i = 0, $exact = false)
    {
        $value = $this->_prepareValue($value, $exact);

        $isUse = false;
        if (!empty($value['sku'])) {
            $select = $this->_conditionSku($select, $value, $exact);
            $isUse  = true;
        }

        if (!empty($value['balance'])) {
            $select = $this->_conditionBalance($select, $value);
            $isUse  = true;
        }

        if (!empty($value['new'])) {
            $select = $this->_conditionNew($select, $value);
            $isUse  = true;
        }

        if (!empty($value['sale'])) {
            $select = $this->_conditionSale($select, $value);
            $isUse  = true;
        }

        if (!empty($value['val']) || !empty($value['val_min']) || !empty($value['val_max']) || !empty($value['range'])) {
            $select = $this->_conditionValue($select, $value);
            $isUse  = true;
        }

        if ($isUse) {
            $select
                ->innerJoin(ZOO_TABLE_JBZOO_SKU . ' AS tSku ON tSku.item_id = tItem.id')
                ->where('tSku.element_id = ?', $elementId);
        }

        return $select;
    }

    /**
     * Set OR element conditions
     * @param JBDatabaseQuery $select
     * @param string $elementId
     * @param string|array $value
     * @param int $i
     * @param bool $exact
     * @return JBDatabaseQuery
     */
    public function conditionOR(JBDatabaseQuery $select, $elementId, $value, $i = 0, $exact = false)
    {
        $value = $this->_prepareValue($value, $exact);

        $itemIds = array();
        if (!empty($value['sku'])) {
            $itemIds[] = $this->_getItemBySku($elementId, $value, $exact);
        }

        if (!empty($value['balance'])) {
            $itemIds[] = $this->_getItemByBalance($elementId);
        }

        if (!empty($value['new'])) {
            $itemIds[] = $this->_getItemByNew($elementId);
        }

        if (!empty($value['sale'])) {
            $itemIds[] = $this->_getItemBySale($elementId);
        }

        if (!empty($value['val']) || !empty($value['val_min']) || !empty($value['val_max']) || !empty($value['range'])) {
            $itemIds[] = $this->_getItemByValue($elementId, $value);
        }

        $where = array();
        foreach ($itemIds as $itemId) {
            $where = array_merge($where, $itemId);
        }

        $where = array_unique($where);

        return 'tItem.id IN (' . implode(',', $where) . ')';
    }

    /**
     * @param JBDatabaseQuery $select
     * @param string $value
     * @param bool $exact
     * @return JBDatabaseQuery
     */
    protected function _conditionSku(JBDatabaseQuery $select, $value, $exact = false)
    {
        if ($exact) {
            $select->where('tSku.sku = ?', $value['sku']);
        } else {
            $select->where($this->_buildLikeBySpaces($value['sku'], 'tSku.sku'));
        }

        return $select;
    }

    /**
     * @param JBDatabaseQuery $select
     * @return JBDatabaseQuery
     */
    protected function _conditionBalance(JBDatabaseQuery $select)
    {
        $select->where('tSku.balance <> 0');

        return $select;
    }

    /**
     * @param JBDatabaseQuery $select
     * @return JBDatabaseQuery
     */
    protected function _conditionNew(JBDatabaseQuery $select)
    {
        $select->where('tSku.is_new = 1');

        return $select;
    }

    /**
     * @param JBDatabaseQuery $select
     * @return JBDatabaseQuery
     */
    protected function _conditionSale(JBDatabaseQuery $select)
    {
        $select->where('tSku.is_sale = 1');

        return $select;
    }

    /**
     * @param JBDatabaseQuery $select
     * @param string $value
     * @return JBDatabaseQuery
     */
    protected function _conditionValue(JBDatabaseQuery $select, $value)
    {
        $jbmoney = $this->app->jbmoney;
        $valType = (int)$value['val_type'];

        if (!empty($value['val'])) {

            $val = $jbmoney->convert($value['currency'], $this->_defaultCurrency, $value['val']);

            $min = floor($val);
            $max = ceil($val);

            if ($valType == 1) {
                $select
                    ->where('tSku.price >= ?', $min)
                    ->where('tSku.price <= ?', $max);

            } else if ($valType == 2) {
                $select
                    ->where('tSku.total >= ?', $min)
                    ->where('tSku.total <= ?', $max);

            } else {
                $select
                    ->where('tSku.price >= ?', $min)
                    ->where('tSku.price <= ?', $max)
                    ->where('tSku.total >= ?', $min)
                    ->where('tSku.total <= ?', $max);
            }
        }

        if (!empty($value['val_min']) || !empty($value['val_max']) || !empty($value['range'])) {

            if (!empty($value['range'])) {
                list($min, $max) = explode('/', $value['range']);
            } else {
                $min = $value['val_min'];
                $max = $value['val_max'];
            }

            $min = floor($jbmoney->convert($value['currency'], $this->_defaultCurrency, $min));
            $max = ceil($jbmoney->convert($value['currency'], $this->_defaultCurrency, $max));

            if ($valType == 1) {
                $select
                    ->where('tSku.price >= ?', $min)
                    ->where('tSku.price <= ?', $max);

            } else if ($valType == 2) {
                $select
                    ->where('tSku.total >= ?', $min)
                    ->where('tSku.total <= ?', $max);

            } else {
                $select
                    ->where('tSku.price >= ?', $min)
                    ->where('tSku.price <= ?', $max)
                    ->where('tSku.total >= ?', $min)
                    ->where('tSku.total <= ?', $max);
            }

        }

        $priceType = (int)$value['price_type'];
        if ($priceType == 1) {
            $select->where('tSku.type = 1');
        } else if ($priceType == 2) {
            $select->where('tSku.type = 2');
        }

        return $select;
    }

    /**
     * Get Itemid list by SKU
     * @param $elementId
     * @param $value
     * @param $exact
     * @return mixed
     */
    protected function _getItemBySku($elementId, $value, $exact = false)
    {
        $select = $this->_getSelect()
            ->select('tSku.item_id')
            ->from(ZOO_TABLE_JBZOO_SKU . ' AS tSku')
            ->where('tSku.element_id = ?', $elementId);

        if ($exact) {
            $select->where('tSku.sku = ?', $value['sku']);
        } else {
            $select->where($this->_buildLikeBySpaces($value['sku'], 'tSku.sku'));
        }

        $result = $this->fetchAll($select);
        $result = $this->_groupBy($result, 'item_id');

        return $result;
    }

    /**
     * Get Itemid list by balance
     * @param $elementId
     * @return mixed
     */
    protected function _getItemByBalance($elementId)
    {
        $select = $this->_getSelect()
            ->select('tSku.item_id')
            ->from(ZOO_TABLE_JBZOO_SKU . ' AS tSku')
            ->where('tSku.element_id = ?', $elementId)
            ->where('tSku.balance <> 0');

        $result = $this->fetchAll($select);
        $result = $this->_groupBy($result, 'item_id');

        return $result;
    }

    /**
     * Get Itemid list by NEW flag
     * @param $elementId
     * @return mixed
     */
    protected function _getItemByNew($elementId)
    {
        $select = $this->_getSelect()
            ->select('tSku.item_id')
            ->from(ZOO_TABLE_JBZOO_SKU . ' AS tSku')
            ->where('tSku.element_id = ?', $elementId)
            ->where('tSku.is_new = 1');

        $result = $this->fetchAll($select);
        $result = $this->_groupBy($result, 'item_id');

        return $result;
    }

    /**
     * Get Itemid list by SALE flag
     * @param $elementId
     * @return mixed
     */
    protected function _getItemBySale($elementId)
    {
        $select = $this->_getSelect()
            ->select('tSku.item_id')
            ->from(ZOO_TABLE_JBZOO_SKU . ' AS tSku')
            ->where('tSku.element_id = ?', $elementId)
            ->where('tSku.is_sale = 1');

        $result = $this->fetchAll($select);
        $result = $this->_groupBy($result, 'item_id');

        return $result;
    }

    /**
     * Get Itemid list by values
     * @param $elementId
     * @param $values
     * @return mixed
     */
    protected function _getItemByValue($elementId, $values)
    {
        $select = $this->_getSelect()
            ->select('tSku.item_id')
            ->from(ZOO_TABLE_JBZOO_SKU . ' AS tSku')
            ->where('tSku.element_id = ?', $elementId);

        $select = $this->_conditionValue($select, $values);

        $result = $this->fetchAll($select);
        $result = $this->_groupBy($result, 'item_id');

        return $result;
    }

}
