<?php
/**
 * JBZoo App is universal Joomla CCK, application for YooTheme Zoo component
 *
 * @package     jbzoo
 * @version     2.x Pro
 * @author      JBZoo App http://jbzoo.com
 * @copyright   Copyright (C) JBZoo.com,  All rights reserved.
 * @license     http://jbzoo.com/license-pro.php JBZoo Licence
 * @coder       Andrey Voytsehovsky <kess@jbzoo.com>
 */

// no direct access
defined('_JEXEC') or die('Restricted access');

// register ElementRepeatable class
App::getInstance('zoo')->loader->register('ElementRepeatable', 'elements:repeatable/repeatable.php');

class ElementJBHyphenTextarea extends ElementRepeatable implements iSubmittable {

    const ROWS = 20;
    const COLS = 60;
    const MAX_HIDDEN_EDITORS = 5;

    const DISPLAY_ALL               = 1;
    const DISPLAY_FIRST             = 2;
    const DISPLAY_ALL_WITHOUT_FIRST = 3;

    const CACHING_DISABLED = 0;
    const CACHING_ENABLED  = 1;
    const CACHING_ENFORCED = 2;

    protected $_elementPath;
    protected $_isHyphenation;
    protected $_caching;


    /**
     * Constructor
     */
    function __construct()
    {
        parent::__construct();
        $this->_getFolder();
        $this->_initLanguage();
    }


    /**
     * Get the folder of the element
     */
    protected function _getFolder()
    {
        $dir = dirname(__FILE__);
        $this->app->path->register($dir, 'jbhyphentextarea');
        $this->_elementPath = $this->app->path->path('jbhyphentextarea:');
    }


    /**
     * Loads main language files
     */
    protected function _initLanguage()
    {
        $language = JFactory::getLanguage();
        $language->load('', $this->_elementPath);
    }


    /**
     * Initializes hyphenation plugin
     */
    protected function _initHyphenation() {
        $this->app->loader->register('phpHypher', 'jbhyphentextarea:hypher/hypher.php');
        $this->hyphenator = new phpHypher($this->_elementPath . '/hypher/hyph_ru_RU.conf');
        $this->hyphenator->set_limits(
            (int)$this->config->get('left_limit', 2),
            (int)$this->config->get('right_limit', 2),
            (int)$this->config->get('length_limit', 4),
            (int)$this->config->get('right_limit_last', 2),
            (int)$this->config->get('left_limit_uc', 2)
        );
    }


    /*
        Function: bindData
            Set data through data array.

        Parameters:
            $data - array

        Returns:
            Void
    */
    public function bindData($data = array()) {

        // set raw input for textarea
        $post = $this->app->request->get('post', JREQUEST_ALLOWRAW);
        foreach ($data as $index => $instance_data) {
            if (isset($post['elements'][$this->identifier][$index]['value'])) {
                $data[$index]['value'] = $post['elements'][$this->identifier][$index]['value'];
            }
        }

        parent::bindData($data);

    }

    /*
        Function: _getSearchData
            Get repeatable elements search data.

        Returns:
            String - Search data
    */
    protected function _getSearchData() {

        // clean html tags
        $value  = JFilterInput::getInstance()->clean($this->get('value', ''));

        return (empty($value) ? null : $value);
    }

    /*
        Function: hasValue
            Override. Checks if the element's value is set.

        Parameters:
            $params - render parameter

        Returns:
            Boolean - true, on success
    */
    public function hasValue($params = array()) {
        $params = $this->app->data->create($params);
        $display = (int)$params->get('display', 1);

        if ($display == self::DISPLAY_ALL) {
            foreach ($this as $self) {
                if ($this->_hasValue($params)) {
                    return true;
                }
            }
        } elseif ($display == self::DISPLAY_FIRST) {
            $this->seek(0);
            return $this->_hasValue($params);
        } elseif ($display == self::DISPLAY_ALL_WITHOUT_FIRST) {
            $this->seek(1);
            while ($this->valid()) {
                if ($this->_hasValue($params)) {
                    return true;
                }
                $this->next();
            }
        }

        return false;
    }

    /*
        Function: render
            Renders the element.

        Parameters:
            $params - render parameter

        Returns:
            String - html
    */
    public function render($params = array()) {

        $params         = $this->app->data->create($params);
        $jplugins       = (int)$this->config->get('jplugins', 1);
        $display        = (int)$params->get('display', 1);
        $this->_caching = (int)$this->config->get('caching', 0);

        if ($this->_isHyphenation = (int)$this->config->get('enable_hyphenation', 1)) {
            $this->_initHyphenation();
        }

        $result = array();

        if ($display == self::DISPLAY_ALL) {
            foreach ($this as $self) {
                $result[] = $this->_render($params);
            }
        } elseif ($display == self::DISPLAY_FIRST) {
            $this->seek(0);
            $result[] = $this->_render($params);
        } elseif ($display == self::DISPLAY_ALL_WITHOUT_FIRST) {
            $this->seek(1);
            while ($this->valid()) {
                $result[] = $this->_render($params);
                $this->next();
            }
        }

        // trigger joomla content plugins
        if ($jplugins) {
            for ($i = 0; $i < count($result); $i++) {
                $result[$i] = $this->app->zoo->triggerContentPlugins($result[$i], array('item_id' => $this->_item->id), 'com_zoo.element.textarea');
            }
        }

        return $this->app->element->applySeparators($params->get('separated_by'), $result);
    }


    /**
     * Renders the element
     * @param array $params
     * @return string
     */
    protected function _render($params = array()) {
        $value = $this->get('value', $this->config->get('default'));

        if ($this->_isHyphenation) {

            if ($this->_caching == self::CACHING_ENABLED || $this->_caching == self::CACHING_ENFORCED) {
                $group = 'jbhyphentext_cache';
                $cacheFlag = false;
                if ($this->_caching == self::CACHING_ENFORCED) {
                    $cacheFlag = true;
                }

                if (!($cachedValue = $this->app->jbcache->get($value, $group, $cacheFlag))) {
                    $cachedValue = $this->hyphenator->hyphenate($value, 'UTF-8');
                    $this->app->jbcache->set($value, $cachedValue, $group, $cacheFlag);
                }

                return $cachedValue;
            } else {
                $value = $this->hyphenator->hyphenate($value, 'UTF-8');

                return $value;
            }
        }

        return $value;
    }


    /*
        Function: loadAssets
            Load elements css/js assets.

        Returns:
            Void
    */
    public function loadAssets() {
        parent::loadAssets();
        if ($this->config->get('repeatable')) {
            $this->app->document->addScript('elements:jbhyphentextarea/textarea.js');
        }
        return $this;
    }

    /*
       Function: edit
           Renders the edit form field.

       Returns:
           String - html
    */
    public function edit() {
        return $this->_edit(array('trusted_mode' => true));
    }

    /*
        Function: renderSubmission
            Renders the element in submission.

       Parameters:
            $params - AppData submission parameters

        Returns:
            String - html
    */
    public function renderSubmission($params = array()) {
        $this->loadAssets();
        return $this->_edit($params);
    }

    protected function _edit($params = array()) {

        $params = $this->app->data->create($params);

        $this->rewind();

        return $this->config->get('repeatable') ? $this->renderLayout($this->getLayout('edit.php'), compact('params')) : $this->_addEditor(0, $this->get('value', $this->config->get('default')), $params->get('trusted_mode', false));

    }

    protected function _addEditor($index, $value = '', $trusted_mode = true) {
        $html[] = '<div class="repeatable-content">';
        if ($trusted_mode) {
            $html[] = $this->app->editor->display($this->getControlName('value', false, $index), htmlspecialchars($value, ENT_QUOTES, 'UTF-8'), null, null, self::COLS, self::ROWS, array('pagebreak', 'readmore', 'article'));
        } else {
            $html[] = $this->app->html->_('control.textarea', $this->getControlName('value', false, $index), $value, 'cols='.self::COLS.' rows='.self::ROWS);
        }
        $html[] = '</div>';
        $html[] = '<span class="delete" title="'.JText::_('Delete Element').'"></span>';
        return implode("\n", $html);
    }

    public function getControlName($name, $array = false, $index = 0) {
        return "elements[{$this->identifier}][{$index}][$name]";
    }

}